function [NODE,ELEM,DEL,KEEP] = equivnode(NODE,ELEM,tol,plotflag)
%EQUIVNODE Performs 1D/2D/3D node equivalencing and renumbers the nodes
% 
%   [NODE,ELEM] = EQUIVNODE(NODE,ELEM), where NODE is an Nn x DIM vector
%   with the Nn nodes and ELEM is an element connectivity matrix or a cell
%   where each entry is an element connectivity vector. Returns renumbered
%   ELEM and equivalenced NODE vector.
%   
%   [NODE,ELEM] = EQUIVNODE(NODE,ELEM,tol,plotflag) where 'tol' specifies
%   the equivalence tolerance, default is 'auto'. Passing 1 or true as the
%   'plotflag' will also plot the equivalenced nodes.
%   
%   [NODE,ELEM,DEL,KEEP] = EQUIVNODE(NODE,ELEM,tol,plotflag) returns the
%   deleted nodes in a vector DEL and the boolean for the kept nodes in
%   KEEP in accordance to the input: true for keep and false for removed.

%   $Revision: 1.3 $  $Date: 2014/06/10 $
%   Tomas Zegard

if nargin<2, error('Not enough input arguments.'), end
dim = size(NODE,2);
if (nargin<3 || isempty(tol) || strcmp(tol,'auto'))
    lims = [min(NODE); max(NODE)];
    length = diff(lims,1);
%     tol = 0.01*mean(length)/size(NODE,1)^(1/dim);
    tol = 1e-3 * nthroot(prod(length)/size(NODE,1),dim); 
end
if nargin<4, plotflag = false; end

fprintf('--- Applying node equivalence with tol = %g\n',tol)
dx = tol / 2;
Nn = size(NODE,1);
MAP = zeros(1,Nn);
MAP(1) = 1;
KEEP = true(Nn,1);
k = 1;
for i=2:Nn
    dist = abs(NODE(1:i-1,:)-repmat(NODE(i,:),i-1,1));
    ind = find(all(dist<dx,2));
    if isempty(ind)
        k = k + 1;
        MAP(i) = k;
    else
        MAP(i) = MAP(ind(1));
        KEEP(i) = false;
    end
end

DEL = NODE(~KEEP,:);
NODE = NODE(KEEP,:);
if iscell(ELEM)
    for i=1:numel(ELEM)
        ELEM{i} = MAP(ELEM{i});
    end
else
    ELEM = MAP(ELEM);
end

if plotflag
    if dim==1
        plot(DEL(:,1),zeros(size(DEL,1),1),'mo')
    elseif dim==2
        plot(DEL(:,1),DEL(:,2),'mo')
    elseif dim==3
        plot3(DEL(:,1),DEL(:,2),DEL(:,3),'mo')
    else
        fprintf('WARNING - Dimensions supported for plotting are 1, 2 and 3D\n')
    end
end
return